[cmdletBinding()]
param($outDir)

function Check-WindowsServer2025 {
  $osname = (get-wmiobject Win32_OperatingSystem).Caption
  if ($osname.Contains("Windows Server 2025")) {
     return $true
  } else {
     return $false
  }
}

function Check-AzureLocal23H2 {
  $osname = (get-wmiobject Win32_OperatingSystem).Caption
  $osver = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").DisplayVersion
  if ($osname.Contains("Azure Stack HCI")) {
     if ($osver -eq "23H2") {
       return $true
     } else {
       return $false
     }
  } else {
     return $false
  }
}

function Check-MicrosoftOsConfigModule {
  $osConfigModule = "Microsoft.OSConfig"
  $moduleExists = Get-InstalledModule -Name $osConfigModule -AllVersions -ErrorAction silentlycontinue

  #New-Item -ItemType Directory $outDir -Force

  if ($moduleExists) {
    return $true
  } else {
    return $false
  }
}


if (Check-WindowsServer2025) {
  Write-Verbose "Check-WindowsServer2025: Windows Server 2025"
} else {
  if (Check-AzureLocal23H2) {
    Write-Verbose "Check-WindowsServer2025: Azure Local 23H2"
  } else {
    Write-Verbose "Check-WindowsServer2025: Unsupported Operating System"
    return $null
    exit
  }
}

if (!(Check-MicrosoftOsConfigModule)) {
  Write-Verbose "Check-MicrosoftOsConfigModule: OsConfig Module dows not exist"
  return $null
  exit
}

$auditScenario = ""
if($outDir -eq $null) {
#  $outDir = $env:TEMP
} else {
  if(!(Test-Path -Path $outDir)) {
    #$outDir = $env:TEMP
    New-Item -ItemType Directory $outDir -Force
  }
} 
if (!($outDir -like "*\")) {
  $outDir = $outDir+"\"
}

$dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
$auditScenario = "Defender/Antivirus"
$logFileName = $outDir+"WindowsBaseLine_AntiVirus_$datetime.csv" 
Write-Verbose $auditScenario
Write-Verbose "LogFile: $logFileName"

$results = (Get-OSConfigDesiredConfiguration -Scenario $auditScenario) 
if($outDir -ne $null) {
  $results |Select Name,Description,DataType,Value, @{ Name = "Status"; Expression={$_.Compliance.Status} }, @{ Name = "Severity"; Expression={$_.Compliance.Severity} }, @{ Name = "Reason"; Expression={$_.Compliance.Reason} }|Export-CSV -Path $logFileName -NoTypeInformation
}

$numOfCompliant = 0
$numOfTotal = 0
foreach ($result in $results) {
  if($result.Compliance.Status -eq "Compliant") {
    $numOfCompliant++
    $numOfTotal++
  } elseif ($result.Compliance.Status -eq "NotCompliant") {
    $numOfTotal++
  }
}
return [math]::Round(($numOfCompliant / $numOfTotal * 100),0)



